import { motion } from 'framer-motion';
import { Sun, Moon, Monitor, Sidebar, PanelTop } from 'lucide-react';
import { useTheme } from '@/contexts/ThemeContext';
import { cn } from '@/lib/utils';
import { Button } from '@/components/ui/button';
import { Label } from '@/components/ui/label';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';

const accentColors = [
  { name: 'Blue', value: 'blue', color: '#3B82F6' },
  { name: 'Purple', value: 'purple', color: '#8B5CF6' },
  { name: 'Green', value: 'green', color: '#10B981' },
  { name: 'Orange', value: 'orange', color: '#F59E0B' },
  { name: 'Pink', value: 'pink', color: '#EC4899' },
  { name: 'Teal', value: 'teal', color: '#14B8A6' },
];

export default function AppearanceSettings() {
  const { theme, setTheme } = useTheme();

  return (
    <div className="p-6 space-y-8 max-w-2xl animate-fade-in">
      <div>
        <h1 className="text-2xl font-bold tracking-tight">Appearance</h1>
        <p className="text-muted-foreground mt-1">
          Customize the look and feel of your CRM
        </p>
      </div>

      {/* Theme Mode */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        className="space-y-4"
      >
        <Label className="text-base font-semibold">Theme Mode</Label>
        <div className="grid grid-cols-3 gap-4">
          {[
            { value: 'light', label: 'Light', icon: Sun },
            { value: 'dark', label: 'Dark', icon: Moon },
            { value: 'system', label: 'System', icon: Monitor },
          ].map(({ value, label, icon: Icon }) => (
            <button
              key={value}
              onClick={() => setTheme({ mode: value as 'light' | 'dark' | 'system' })}
              className={cn(
                'flex flex-col items-center gap-3 rounded-xl border-2 p-4 transition-all',
                theme.mode === value
                  ? 'border-primary bg-primary/5'
                  : 'border-border hover:border-primary/50'
              )}
            >
              <Icon className={cn(
                'h-6 w-6',
                theme.mode === value ? 'text-primary' : 'text-muted-foreground'
              )} />
              <span className={cn(
                'text-sm font-medium',
                theme.mode === value ? 'text-primary' : 'text-muted-foreground'
              )}>
                {label}
              </span>
            </button>
          ))}
        </div>
      </motion.div>

      {/* Layout Mode */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.1 }}
        className="space-y-4"
      >
        <Label className="text-base font-semibold">Navigation Layout</Label>
        <div className="grid grid-cols-2 gap-4">
          {[
            { value: 'sidebar', label: 'Sidebar', icon: Sidebar },
            { value: 'topbar', label: 'Topbar', icon: PanelTop },
          ].map(({ value, label, icon: Icon }) => (
            <button
              key={value}
              onClick={() => setTheme({ layoutMode: value as 'sidebar' | 'topbar' })}
              className={cn(
                'flex flex-col items-center gap-3 rounded-xl border-2 p-4 transition-all',
                theme.layoutMode === value
                  ? 'border-primary bg-primary/5'
                  : 'border-border hover:border-primary/50'
              )}
            >
              <Icon className={cn(
                'h-6 w-6',
                theme.layoutMode === value ? 'text-primary' : 'text-muted-foreground'
              )} />
              <span className={cn(
                'text-sm font-medium',
                theme.layoutMode === value ? 'text-primary' : 'text-muted-foreground'
              )}>
                {label}
              </span>
            </button>
          ))}
        </div>
      </motion.div>

      {/* Accent Color */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.2 }}
        className="space-y-4"
      >
        <Label className="text-base font-semibold">Accent Color</Label>
        <div className="flex gap-3">
          {accentColors.map((accent) => (
            <button
              key={accent.value}
              onClick={() => setTheme({ accentColor: accent.value })}
              className={cn(
                'h-10 w-10 rounded-full transition-transform',
                theme.accentColor === accent.value && 'ring-2 ring-offset-2 ring-primary scale-110'
              )}
              style={{ backgroundColor: accent.color }}
              title={accent.name}
            />
          ))}
        </div>
      </motion.div>

      {/* Density */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.3 }}
        className="space-y-4"
      >
        <Label className="text-base font-semibold">Density</Label>
        <RadioGroup
          value={theme.density}
          onValueChange={(value) => setTheme({ density: value as 'comfortable' | 'compact' })}
          className="flex gap-4"
        >
          <div className="flex items-center space-x-2">
            <RadioGroupItem value="comfortable" id="comfortable" />
            <Label htmlFor="comfortable" className="cursor-pointer">Comfortable</Label>
          </div>
          <div className="flex items-center space-x-2">
            <RadioGroupItem value="compact" id="compact" />
            <Label htmlFor="compact" className="cursor-pointer">Compact</Label>
          </div>
        </RadioGroup>
      </motion.div>
    </div>
  );
}
