import { useState, useMemo } from 'react';
import { format, differenceInDays, isToday, isYesterday } from 'date-fns';
import { motion, AnimatePresence } from 'framer-motion';
import {
  Trash2,
  RotateCcw,
  AlertTriangle,
  Loader2,
  Calendar,
  MapPin,
  Search,
  Check,
  X,
  Archive,
  Clock,
  Filter,
  ChevronDown,
  Info,
} from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent } from '@/components/ui/card';
import { Checkbox } from '@/components/ui/checkbox';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from '@/components/ui/alert-dialog';
import {
  Tooltip,
  TooltipContent,
  TooltipProvider,
  TooltipTrigger,
} from '@/components/ui/tooltip';
import { useTrashedContacts } from '@/hooks/useTrash';
import { toast } from 'sonner';
import { cn } from '@/lib/utils';

type SortOption = 'date-desc' | 'date-asc' | 'name-asc' | 'name-desc';

export default function Trash() {
  const {
    trashedContacts,
    isLoading,
    restore,
    permanentDelete,
    emptyTrash,
    isRestoring,
    isDeleting,
    isEmptyingTrash,
  } = useTrashedContacts();

  const [searchQuery, setSearchQuery] = useState('');
  const [sortBy, setSortBy] = useState<SortOption>('date-desc');
  const [selectedIds, setSelectedIds] = useState<Set<string>>(new Set());
  const [deleteId, setDeleteId] = useState<string | null>(null);
  const [showEmptyTrashDialog, setShowEmptyTrashDialog] = useState(false);
  const [showBulkDeleteDialog, setShowBulkDeleteDialog] = useState(false);
  const [filterWarning, setFilterWarning] = useState(false);

  const AUTO_DELETE_DAYS = 30;

  const getInitials = (firstName: string, lastName: string) => {
    return `${firstName[0] ?? ''}${lastName[0] ?? ''}`.toUpperCase();
  };

  const getDaysUntilAutoDelete = (deletedAt: string) => {
    const daysSinceDelete = differenceInDays(new Date(), new Date(deletedAt));
    return AUTO_DELETE_DAYS - daysSinceDelete;
  };

  const formatDeletedDate = (deletedAt: string) => {
    const date = new Date(deletedAt);
    if (isToday(date)) return 'Today';
    if (isYesterday(date)) return 'Yesterday';
    return format(date, 'MMM d, yyyy');
  };

  const groupContactsByDate = (contacts: any[]) => {
    const groups: Record<string, any[]> = {};
    contacts.forEach((contact) => {
      const key = formatDeletedDate(contact.deleted_at);
      if (!groups[key]) groups[key] = [];
      groups[key].push(contact);
    });
    return groups;
  };

  // Filter and sort contacts
  const filteredAndSortedContacts = useMemo(() => {
    let filtered = trashedContacts.filter((contact) => {
      const searchLower = searchQuery.toLowerCase();
      const matchesSearch =
        contact.first_name.toLowerCase().includes(searchLower) ||
        contact.last_name.toLowerCase().includes(searchLower) ||
        contact.email?.toLowerCase().includes(searchLower) ||
        contact.phone?.toLowerCase().includes(searchLower);

      if (filterWarning) {
        const daysLeft = getDaysUntilAutoDelete(contact.deleted_at);
        return matchesSearch && daysLeft <= 7;
      }

      return matchesSearch;
    });

    // Sort
    filtered.sort((a, b) => {
      switch (sortBy) {
        case 'date-desc':
          return new Date(b.deleted_at).getTime() - new Date(a.deleted_at).getTime();
        case 'date-asc':
          return new Date(a.deleted_at).getTime() - new Date(b.deleted_at).getTime();
        case 'name-asc':
          return `${a.first_name} ${a.last_name}`.localeCompare(
            `${b.first_name} ${b.last_name}`
          );
        case 'name-desc':
          return `${b.first_name} ${b.last_name}`.localeCompare(
            `${a.first_name} ${a.last_name}`
          );
        default:
          return 0;
      }
    });

    return filtered;
  }, [trashedContacts, searchQuery, sortBy, filterWarning]);

  const groupedContacts = groupContactsByDate(filteredAndSortedContacts);

  const warningContacts = trashedContacts.filter(
    (c) => getDaysUntilAutoDelete(c.deleted_at) <= 7
  );

  const handleRestore = async (id: string) => {
    await restore(id);
    setSelectedIds((prev) => {
      const next = new Set(prev);
      next.delete(id);
      return next;
    });
    toast.success('Contact restored successfully', {
      description: 'The contact has been moved back to your contacts list',
    });
  };

  const handleBulkRestore = async () => {
    const promises = Array.from(selectedIds).map((id) => restore(id));
    await Promise.all(promises);
    toast.success(`${selectedIds.size} contacts restored`, {
      description: 'All selected contacts have been restored',
    });
    setSelectedIds(new Set());
  };

  const handlePermanentDelete = async () => {
    if (deleteId) {
      await permanentDelete(deleteId);
      setDeleteId(null);
      toast.success('Contact permanently deleted');
    }
  };

  const handleBulkDelete = async () => {
    const promises = Array.from(selectedIds).map((id) => permanentDelete(id));
    await Promise.all(promises);
    toast.success(`${selectedIds.size} contacts permanently deleted`);
    setSelectedIds(new Set());
    setShowBulkDeleteDialog(false);
  };

  const handleEmptyTrash = async () => {
    await emptyTrash();
    setShowEmptyTrashDialog(false);
    setSelectedIds(new Set());
    toast.success('Trash emptied successfully');
  };

  const toggleSelectAll = () => {
    if (selectedIds.size === filteredAndSortedContacts.length) {
      setSelectedIds(new Set());
    } else {
      setSelectedIds(new Set(filteredAndSortedContacts.map((c) => c.id)));
    }
  };

  const toggleSelect = (id: string) => {
    setSelectedIds((prev) => {
      const next = new Set(prev);
      if (next.has(id)) {
        next.delete(id);
      } else {
        next.add(id);
      }
      return next;
    });
  };

  if (isLoading) {
    return (
      <div className="p-6 space-y-6">
        <div className="space-y-4">
          <div className="h-8 w-48 bg-muted animate-pulse rounded" />
          <div className="h-4 w-96 bg-muted animate-pulse rounded" />
        </div>
        <div className="space-y-3">
          {[1, 2, 3].map((i) => (
            <Card key={i}>
              <CardContent className="p-4">
                <div className="flex items-center gap-4">
                  <div className="h-12 w-12 rounded-full bg-muted animate-pulse" />
                  <div className="flex-1 space-y-2">
                    <div className="h-4 w-48 bg-muted animate-pulse rounded" />
                    <div className="h-3 w-64 bg-muted animate-pulse rounded" />
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  return (
    <TooltipProvider>
      <div className="p-6 space-y-6 animate-fade-in">
        {/* Header */}
        <div className="flex items-start justify-between gap-4">
          <div>
            <h1 className="text-3xl font-bold tracking-tight">Trash</h1>
            <p className="text-muted-foreground mt-1">
              Deleted contacts are permanently removed after {AUTO_DELETE_DAYS} days
            </p>
          </div>
          {trashedContacts.length > 0 && (
            <Button
              variant="destructive"
              onClick={() => setShowEmptyTrashDialog(true)}
              disabled={isEmptyingTrash}
              className="gap-2"
            >
              {isEmptyingTrash ? (
                <Loader2 className="h-4 w-4 animate-spin" />
              ) : (
                <Trash2 className="h-4 w-4" />
              )}
              Empty Trash
            </Button>
          )}
        </div>

        {/* Stats & Warnings */}
        {trashedContacts.length > 0 && (
          <div className="grid gap-4 md:grid-cols-3">
            <Card>
              <CardContent className="p-4">
                <div className="flex items-center gap-3">
                  <div className="p-2 bg-primary/10 rounded-lg">
                    <Archive className="h-5 w-5 text-primary" />
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground">Total Items</p>
                    <p className="text-2xl font-bold">{trashedContacts.length}</p>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardContent className="p-4">
                <div className="flex items-center gap-3">
                  <div className="p-2 bg-blue-500/10 rounded-lg">
                    <Calendar className="h-5 w-5 text-blue-500" />
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground">Selected</p>
                    <p className="text-2xl font-bold">{selectedIds.size}</p>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card
              className={cn(
                'border-2',
                warningContacts.length > 0 && 'border-orange-500/50 bg-orange-500/5'
              )}
            >
              <CardContent className="p-4">
                <div className="flex items-center gap-3">
                  <div className="p-2 bg-orange-500/10 rounded-lg">
                    <Clock className="h-5 w-5 text-orange-500" />
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground">Auto-Delete Soon</p>
                    <p className="text-2xl font-bold text-orange-500">
                      {warningContacts.length}
                    </p>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        )}

        {/* Warning Banner */}
        {warningContacts.length > 0 && (
          <motion.div
            initial={{ opacity: 0, y: -10 }}
            animate={{ opacity: 1, y: 0 }}
          >
            <Card className="border-orange-500/50 bg-orange-500/5">
              <CardContent className="p-4">
                <div className="flex items-start gap-3">
                  <AlertTriangle className="h-5 w-5 text-orange-500 mt-0.5 flex-shrink-0" />
                  <div className="flex-1">
                    <p className="font-medium text-orange-900 dark:text-orange-100">
                      {warningContacts.length} contact{warningContacts.length > 1 ? 's' : ''} will
                      be permanently deleted soon
                    </p>
                    <p className="text-sm text-orange-700 dark:text-orange-300 mt-1">
                      These contacts will be automatically deleted within 7 days. Restore them now
                      to keep them.
                    </p>
                  </div>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setFilterWarning(!filterWarning)}
                    className="border-orange-500 text-orange-700 hover:bg-orange-500/10"
                  >
                    {filterWarning ? 'Show All' : 'Show Warning Items'}
                  </Button>
                </div>
              </CardContent>
            </Card>
          </motion.div>
        )}

        {trashedContacts.length === 0 ? (
          <Card className="border-dashed">
            <CardContent className="py-24 text-center">
              <motion.div
                initial={{ scale: 0.9, opacity: 0 }}
                animate={{ scale: 1, opacity: 1 }}
                transition={{ duration: 0.3 }}
              >
                <div className="mx-auto w-16 h-16 rounded-full bg-muted flex items-center justify-center mb-4">
                  <Trash2 className="h-8 w-8 text-muted-foreground" />
                </div>
                <h3 className="text-xl font-semibold text-foreground mb-2">
                  Trash is empty
                </h3>
                <p className="text-muted-foreground max-w-md mx-auto">
                  Deleted contacts will appear here and can be restored within {AUTO_DELETE_DAYS}{' '}
                  days before being permanently removed.
                </p>
              </motion.div>
            </CardContent>
          </Card>
        ) : (
          <>
            {/* Controls */}
            <div className="flex flex-col sm:flex-row gap-3">
              <div className="relative flex-1">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Search deleted contacts..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pl-9"
                />
              </div>

              <Select value={sortBy} onValueChange={(v) => setSortBy(v as SortOption)}>
                <SelectTrigger className="w-full sm:w-48">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="date-desc">Newest First</SelectItem>
                  <SelectItem value="date-asc">Oldest First</SelectItem>
                  <SelectItem value="name-asc">Name A-Z</SelectItem>
                  <SelectItem value="name-desc">Name Z-A</SelectItem>
                </SelectContent>
              </Select>

              {selectedIds.size > 0 && (
                <div className="flex gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={handleBulkRestore}
                    disabled={isRestoring}
                    className="gap-2"
                  >
                    <RotateCcw className="h-4 w-4" />
                    Restore ({selectedIds.size})
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setShowBulkDeleteDialog(true)}
                    disabled={isDeleting}
                    className="gap-2 text-destructive hover:text-destructive"
                  >
                    <Trash2 className="h-4 w-4" />
                    Delete ({selectedIds.size})
                  </Button>
                </div>
              )}
            </div>

            {/* Select All */}
            {filteredAndSortedContacts.length > 0 && (
              <div className="flex items-center gap-2 px-2">
                <Checkbox
                  checked={selectedIds.size === filteredAndSortedContacts.length}
                  onCheckedChange={toggleSelectAll}
                  id="select-all"
                />
                <label
                  htmlFor="select-all"
                  className="text-sm font-medium cursor-pointer select-none"
                >
                  Select all ({filteredAndSortedContacts.length})
                </label>
              </div>
            )}

            {/* Contact List - Grouped by Date */}
            <div className="space-y-6">
              <AnimatePresence mode="popLayout">
                {Object.entries(groupedContacts).map(([dateGroup, contacts]) => (
                  <motion.div
                    key={dateGroup}
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1 }}
                    exit={{ opacity: 0 }}
                    className="space-y-3"
                  >
                    <div className="flex items-center gap-2 px-2">
                      <h3 className="text-sm font-semibold text-muted-foreground uppercase tracking-wider">
                        {dateGroup}
                      </h3>
                      <div className="flex-1 h-px bg-border" />
                      <span className="text-xs text-muted-foreground">
                        {contacts.length} item{contacts.length > 1 ? 's' : ''}
                      </span>
                    </div>

                    <div className="grid gap-3">
                      {contacts.map((contact, index) => {
                        const daysLeft = getDaysUntilAutoDelete(contact.deleted_at);
                        const isNearExpiry = daysLeft <= 7;

                        return (
                          <motion.div
                            key={contact.id}
                            layout
                            initial={{ opacity: 0, y: 10 }}
                            animate={{ opacity: 1, y: 0 }}
                            exit={{ opacity: 0, scale: 0.95 }}
                            transition={{ delay: index * 0.03 }}
                          >
                            <Card
                              className={cn(
                                'hover:bg-accent/50 transition-all group',
                                selectedIds.has(contact.id) && 'ring-2 ring-primary',
                                isNearExpiry && 'border-orange-500/30'
                              )}
                            >
                              <CardContent className="p-4">
                                <div className="flex items-center gap-4">
                                  <Checkbox
                                    checked={selectedIds.has(contact.id)}
                                    onCheckedChange={() => toggleSelect(contact.id)}
                                    className="mt-1"
                                  />

                                  <Avatar className="h-12 w-12 ring-2 ring-background">
                                    {contact.avatar_url ? (
                                      <AvatarImage src={contact.avatar_url} />
                                    ) : null}
                                    <AvatarFallback className="bg-gradient-to-br from-primary/20 to-primary/5 text-primary font-semibold">
                                      {getInitials(contact.first_name, contact.last_name)}
                                    </AvatarFallback>
                                  </Avatar>

                                  <div className="flex-1 min-w-0">
                                    <div className="flex items-center gap-2 mb-1">
                                      <h3 className="font-semibold text-foreground truncate">
                                        {contact.first_name} {contact.last_name}
                                      </h3>
                                      {contact.tags?.slice(0, 2).map((tag) => (
                                        <Badge
                                          key={tag.id}
                                          variant="outline"
                                          className="text-xs"
                                          style={{
                                            borderColor: tag.color,
                                            color: tag.color,
                                          }}
                                        >
                                          {tag.name}
                                        </Badge>
                                      ))}
                                      {isNearExpiry && (
                                        <Tooltip>
                                          <TooltipTrigger>
                                            <Badge
                                              variant="outline"
                                              className="text-xs border-orange-500 text-orange-500"
                                            >
                                              <Clock className="h-3 w-3 mr-1" />
                                              {daysLeft}d left
                                            </Badge>
                                          </TooltipTrigger>
                                          <TooltipContent>
                                            <p>
                                              Will be permanently deleted in {daysLeft} day
                                              {daysLeft !== 1 ? 's' : ''}
                                            </p>
                                          </TooltipContent>
                                        </Tooltip>
                                      )}
                                    </div>

                                    <p className="text-sm text-muted-foreground truncate mb-2">
                                      {contact.email}
                                    </p>

                                    <div className="flex flex-wrap items-center gap-3 text-xs text-muted-foreground">
                                      <span className="flex items-center gap-1">
                                        <Calendar className="h-3 w-3" />
                                        Deleted{' '}
                                        {contact.deleted_at
                                          ? format(new Date(contact.deleted_at), 'MMM d, HH:mm')
                                          : 'Unknown'}
                                      </span>
                                      {contact.deleted_from && (
                                        <span className="flex items-center gap-1">
                                          <MapPin className="h-3 w-3" />
                                          {contact.deleted_from}
                                        </span>
                                      )}
                                    </div>
                                  </div>

                                  <div className="flex items-center gap-2 opacity-0 group-hover:opacity-100 transition-opacity">
                                    <Tooltip>
                                      <TooltipTrigger asChild>
                                        <Button
                                          variant="outline"
                                          size="sm"
                                          onClick={() => handleRestore(contact.id)}
                                          disabled={isRestoring}
                                          className="gap-2"
                                        >
                                          <RotateCcw className="h-4 w-4" />
                                          Restore
                                        </Button>
                                      </TooltipTrigger>
                                      <TooltipContent>
                                        <p>Restore to contacts</p>
                                      </TooltipContent>
                                    </Tooltip>

                                    <Tooltip>
                                      <TooltipTrigger asChild>
                                        <Button
                                          variant="ghost"
                                          size="icon"
                                          onClick={() => setDeleteId(contact.id)}
                                          disabled={isDeleting}
                                          className="text-destructive hover:text-destructive hover:bg-destructive/10"
                                        >
                                          <Trash2 className="h-4 w-4" />
                                        </Button>
                                      </TooltipTrigger>
                                      <TooltipContent>
                                        <p>Delete permanently</p>
                                      </TooltipContent>
                                    </Tooltip>
                                  </div>
                                </div>
                              </CardContent>
                            </Card>
                          </motion.div>
                        );
                      })}
                    </div>
                  </motion.div>
                ))}
              </AnimatePresence>
            </div>

            {/* No Results */}
            {filteredAndSortedContacts.length === 0 && (
              <Card className="border-dashed">
                <CardContent className="py-16 text-center">
                  <Search className="h-12 w-12 mx-auto text-muted-foreground mb-4 opacity-50" />
                  <h3 className="text-lg font-medium text-foreground mb-2">No results found</h3>
                  <p className="text-muted-foreground">
                    {filterWarning
                      ? 'No contacts are near auto-deletion'
                      : 'Try adjusting your search query'}
                  </p>
                  {(searchQuery || filterWarning) && (
                    <Button
                      variant="outline"
                      className="mt-4"
                      onClick={() => {
                        setSearchQuery('');
                        setFilterWarning(false);
                      }}
                    >
                      Clear filters
                    </Button>
                  )}
                </CardContent>
              </Card>
            )}
          </>
        )}

        {/* Permanent Delete Single Confirmation */}
        <AlertDialog open={!!deleteId} onOpenChange={() => setDeleteId(null)}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle className="flex items-center gap-2">
                <AlertTriangle className="h-5 w-5 text-destructive" />
                Permanently Delete Contact
              </AlertDialogTitle>
              <AlertDialogDescription className="space-y-2">
                <p>This action cannot be undone. This will permanently delete the contact.</p>
                <div className="bg-muted p-3 rounded-md mt-3">
                  <p className="text-sm font-medium text-foreground mb-1">
                    The following will be deleted:
                  </p>
                  <ul className="text-sm text-muted-foreground space-y-1 list-disc list-inside">
                    <li>Contact information and profile</li>
                    <li>All attachments and files</li>
                    <li>Activity history and notes</li>
                    <li>Associated communications</li>
                  </ul>
                </div>
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Cancel</AlertDialogCancel>
              <AlertDialogAction
                onClick={handlePermanentDelete}
                className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
                disabled={isDeleting}
              >
                {isDeleting ? (
                  <>
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                    Deleting...
                  </>
                ) : (
                  'Delete Permanently'
                )}
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>

        {/* Bulk Delete Confirmation */}
        <AlertDialog open={showBulkDeleteDialog} onOpenChange={setShowBulkDeleteDialog}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle className="flex items-center gap-2">
                <AlertTriangle className="h-5 w-5 text-destructive" />
                Delete {selectedIds.size} Contact{selectedIds.size > 1 ? 's' : ''}
              </AlertDialogTitle>
              <AlertDialogDescription>
                This will permanently delete {selectedIds.size} selected contact
                {selectedIds.size > 1 ? 's' : ''} along with all their data, attachments, and
                files. This action cannot be undone.
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Cancel</AlertDialogCancel>
              <AlertDialogAction
                onClick={handleBulkDelete}
                className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
                disabled={isDeleting}
              >
                {isDeleting ? (
                  <>
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                    Deleting...
                  </>
                ) : (
                  `Delete ${selectedIds.size} Contact${selectedIds.size > 1 ? 's' : ''}`
                )}
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>

        {/* Empty Trash Confirmation */}
        <AlertDialog open={showEmptyTrashDialog} onOpenChange={setShowEmptyTrashDialog}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle className="flex items-center gap-2">
                <AlertTriangle className="h-5 w-5 text-destructive" />
                Empty Entire Trash
              </AlertDialogTitle>
              <AlertDialogDescription className="space-y-2">
                <p>
                  This will permanently delete all <strong>{trashedContacts.length}</strong>{' '}
                  contacts in the trash along with all their data. This action cannot be undone.
                </p>
                <div className="bg-destructive/10 border border-destructive/20 p-3 rounded-md mt-3">
                  <p className="text-sm font-medium text-destructive flex items-center gap-2">
                    <Info className="h-4 w-4" />
                    Warning: This is irreversible
                  </p>
                  <p className="text-sm text-muted-foreground mt-1">
                    All contact data, files, and history will be permanently lost.
                  </p>
                </div>
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Cancel</AlertDialogCancel>
              <AlertDialogAction
                onClick={handleEmptyTrash}
                className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
                disabled={isEmptyingTrash}
              >
                {isEmptyingTrash ? (
                  <>
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                    Emptying...
                  </>
                ) : (
                  'Empty Trash'
                )}
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>
      </div>
    </TooltipProvider>
  );
}