import React, { useState } from 'react';
import { useParams, useNavigate, Link } from 'react-router-dom';
import { format } from 'date-fns';
import {
    ArrowLeft,
    Edit,
    Trash2,
    Building2,
    MapPin,
    PoundSterling,
    User,
    Calendar,
    FileText,
    Image,
    ExternalLink,
    AlertTriangle,
    Activity,
    MessageSquare,
    Home,
    Bed,
    Bath,
    Sofa,
    Shield,
    Link2,
    MoreHorizontal,
    Share2,
    Copy,
    Check,
    Clock,
    TrendingUp,
    Zap,
    Globe,
    Key,
    Flame,
    Gauge,
    FileCheck,
    AlertCircle,
    ChevronRight,
    Sparkles,
} from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuSeparator,
    DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import {
    AlertDialog,
    AlertDialogAction,
    AlertDialogCancel,
    AlertDialogContent,
    AlertDialogDescription,
    AlertDialogFooter,
    AlertDialogHeader,
    AlertDialogTitle,
} from '@/components/ui/alert-dialog';
import { Skeleton } from '@/components/ui/skeleton';
import { Separator } from '@/components/ui/separator';
import { ScrollArea, ScrollBar } from '@/components/ui/scroll-area';
import { Avatar, AvatarFallback } from '@/components/ui/avatar';
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from '@/components/ui/tooltip';
import { cn } from '@/lib/utils';
import { useProperty, useDeleteProperty } from '@/hooks/useProperties';
import { PropertyFormDialog } from '@/components/properties/PropertyFormDialog';
import { PropertyImagesTab } from '@/components/properties/PropertyImagesTab';
import { PropertyDocumentsTab } from '@/components/properties/PropertyDocumentsTab';
import { PropertyActivitiesTab } from '@/components/properties/PropertyActivitiesTab';
import { PropertyNotesTab } from '@/components/properties/PropertyNotesTab';
import { PropertyTenanciesTab } from '@/components/properties/PropertyTenanciesTab';

export default function PropertyDetail() {
    const { id } = useParams<{ id: string }>();
    const navigate = useNavigate();
    const { data: property, isLoading } = useProperty(id);
    const deleteProperty = useDeleteProperty();

    const [editOpen, setEditOpen] = useState(false);
    const [deleteOpen, setDeleteOpen] = useState(false);
    const [copied, setCopied] = useState(false);

    const handleDelete = async () => {
        if (id) {
            await deleteProperty.mutateAsync(id);
            navigate('/properties');
        }
    };

    const handleCopyAddress = () => {
        const address = [
            property?.address_line_1,
            property?.address_line_2,
            property?.town_city,
            property?.county,
            property?.postcode,
        ]
            .filter(Boolean)
            .join(', ');
        navigator.clipboard.writeText(address);
        setCopied(true);
        setTimeout(() => setCopied(false), 2000);
    };

    const getStatusConfig = (status: string | null) => {
        switch (status) {
            case 'available':
                return {
                    color: 'bg-emerald-500/10 text-emerald-600 border-emerald-500/20',
                    icon: Check,
                    label: 'Available',
                };
            case 'let':
                return {
                    color: 'bg-blue-500/10 text-blue-600 border-blue-500/20',
                    icon: Key,
                    label: 'Let',
                };
            case 'under_offer':
                return {
                    color: 'bg-amber-500/10 text-amber-600 border-amber-500/20',
                    icon: Clock,
                    label: 'Under Offer',
                };
            case 'sold':
                return {
                    color: 'bg-purple-500/10 text-purple-600 border-purple-500/20',
                    icon: TrendingUp,
                    label: 'Sold',
                };
            default:
                return {
                    color: 'bg-muted text-muted-foreground',
                    icon: AlertCircle,
                    label: 'Unknown',
                };
        }
    };

    const isComplianceExpiring = (date: string | null) => {
        if (!date) return false;
        const expiryDate = new Date(date);
        const thirtyDaysFromNow = new Date();
        thirtyDaysFromNow.setDate(thirtyDaysFromNow.getDate() + 30);
        return expiryDate <= thirtyDaysFromNow;
    };

    const isComplianceExpired = (date: string | null) => {
        if (!date) return false;
        return new Date(date) < new Date();
    };

    if (isLoading) {
        return (
            <div className="min-h-screen bg-gradient-to-br from-background via-background to-muted/20">
                <div className="max-w-7xl mx-auto p-6 space-y-6">
                    <div className="flex items-center gap-4">
                        <Skeleton className="h-10 w-10 rounded-xl" />
                        <Skeleton className="h-8 w-64" />
                    </div>
                    <Skeleton className="h-[200px] w-full rounded-2xl" />
                    <div className="grid grid-cols-4 gap-4">
                        {[...Array(4)].map((_, i) => (
                            <Skeleton key={i} className="h-[100px] rounded-xl" />
                        ))}
                    </div>
                    <Skeleton className="h-[400px] w-full rounded-2xl" />
                </div>
            </div>
        );
    }

    if (!property) {
        return (
            <div className="min-h-screen bg-gradient-to-br from-background via-background to-muted/20 flex items-center justify-center">
                <div className="text-center space-y-4">
                    <div className="w-20 h-20 rounded-2xl bg-muted/50 flex items-center justify-center mx-auto">
                        <Building2 className="h-10 w-10 text-muted-foreground" />
                    </div>
                    <h2 className="text-2xl font-semibold">Property not found</h2>
                    <p className="text-muted-foreground">
                        The property you're looking for doesn't exist or has been removed.
                    </p>
                    <Button onClick={() => navigate('/properties')} className="mt-4">
                        <ArrowLeft className="mr-2 h-4 w-4" />
                        Back to Properties
                    </Button>
                </div>
            </div>
        );
    }

    const statusConfig = getStatusConfig(property.status);
    const StatusIcon = statusConfig.icon;

    const DetailItem = ({
        label,
        value,
        icon: Icon,
        className,
    }: {
        label: string;
        value: React.ReactNode;
        icon?: React.ElementType;
        className?: string;
    }) => (
        <div className={cn('space-y-1', className)}>
            <p className="text-xs font-medium text-muted-foreground uppercase tracking-wider flex items-center gap-1.5">
                {Icon && <Icon className="h-3 w-3" />}
                {label}
            </p>
            <p className="text-sm font-medium">{value || <span className="text-muted-foreground">—</span>}</p>
        </div>
    );

    const ComplianceItem = ({
        label,
        date,
        icon: Icon,
    }: {
        label: string;
        date: string | null;
        icon: React.ElementType;
    }) => {
        const expired = isComplianceExpired(date);
        const expiring = isComplianceExpiring(date);

        return (
            <div
                className={cn(
                    'flex items-center justify-between p-4 rounded-xl border transition-colors',
                    expired
                        ? 'bg-red-500/5 border-red-500/20'
                        : expiring
                            ? 'bg-amber-500/5 border-amber-500/20'
                            : 'bg-muted/30 border-transparent'
                )}
            >
                <div className="flex items-center gap-3">
                    <div
                        className={cn(
                            'p-2 rounded-lg',
                            expired
                                ? 'bg-red-500/10 text-red-600'
                                : expiring
                                    ? 'bg-amber-500/10 text-amber-600'
                                    : 'bg-muted text-muted-foreground'
                        )}
                    >
                        <Icon className="h-4 w-4" />
                    </div>
                    <div>
                        <p className="font-medium text-sm">{label}</p>
                        <p
                            className={cn(
                                'text-xs',
                                expired
                                    ? 'text-red-600'
                                    : expiring
                                        ? 'text-amber-600'
                                        : 'text-muted-foreground'
                            )}
                        >
                            {date ? format(new Date(date), 'dd MMM yyyy') : 'Not set'}
                        </p>
                    </div>
                </div>
                {(expired || expiring) && (
                    <Badge
                        variant="outline"
                        className={cn(
                            'text-xs',
                            expired
                                ? 'border-red-500/20 text-red-600 bg-red-500/10'
                                : 'border-amber-500/20 text-amber-600 bg-amber-500/10'
                        )}
                    >
                        {expired ? 'Expired' : 'Expiring Soon'}
                    </Badge>
                )}
            </div>
        );
    };

    return (
        <div className="min-h-screen bg-gradient-to-br from-background via-background to-muted/20">
            <div className="max-w-7xl mx-auto">
                {/* Header */}
                <div className="sticky top-0 z-40 bg-background/80 backdrop-blur-xl border-b">
                    <div className="px-6 py-4">
                        <div className="flex items-center justify-between">
                            <div className="flex items-center gap-4">
                                <Button
                                    variant="ghost"
                                    size="icon"
                                    onClick={() => navigate('/properties')}
                                    className="rounded-xl hover:bg-muted"
                                >
                                    <ArrowLeft className="h-4 w-4" />
                                </Button>
                                <div className="flex items-center gap-3">
                                    <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-primary/20 to-primary/5 flex items-center justify-center">
                                        <Building2 className="h-5 w-5 text-primary" />
                                    </div>
                                    <div>
                                        <h1 className="text-lg font-semibold line-clamp-1">
                                            {property.display_address || property.reference || 'Untitled Property'}
                                        </h1>
                                        <p className="text-sm text-muted-foreground">
                                            {property.reference && `Ref: ${property.reference}`}
                                        </p>
                                    </div>
                                </div>
                            </div>

                            <div className="flex items-center gap-2">
                                <TooltipProvider>
                                    <Tooltip>
                                        <TooltipTrigger asChild>
                                            <Button
                                                variant="ghost"
                                                size="icon"
                                                className="rounded-xl"
                                                onClick={handleCopyAddress}
                                            >
                                                {copied ? (
                                                    <Check className="h-4 w-4 text-green-500" />
                                                ) : (
                                                    <Copy className="h-4 w-4" />
                                                )}
                                            </Button>
                                        </TooltipTrigger>
                                        <TooltipContent>Copy address</TooltipContent>
                                    </Tooltip>
                                </TooltipProvider>

                                <Button variant="ghost" size="icon" className="rounded-xl">
                                    <Share2 className="h-4 w-4" />
                                </Button>

                                <Separator orientation="vertical" className="h-6" />

                                <Button
                                    variant="outline"
                                    onClick={() => setEditOpen(true)}
                                    className="rounded-xl"
                                >
                                    <Edit className="mr-2 h-4 w-4" />
                                    Edit
                                </Button>

                                <DropdownMenu>
                                    <DropdownMenuTrigger asChild>
                                        <Button variant="ghost" size="icon" className="rounded-xl">
                                            <MoreHorizontal className="h-4 w-4" />
                                        </Button>
                                    </DropdownMenuTrigger>
                                    <DropdownMenuContent align="end" className="w-48">
                                        <DropdownMenuItem>
                                            <Copy className="mr-2 h-4 w-4" />
                                            Duplicate
                                        </DropdownMenuItem>
                                        <DropdownMenuItem>
                                            <FileText className="mr-2 h-4 w-4" />
                                            Export PDF
                                        </DropdownMenuItem>
                                        <DropdownMenuSeparator />
                                        <DropdownMenuItem
                                            className="text-destructive focus:text-destructive"
                                            onClick={() => setDeleteOpen(true)}
                                        >
                                            <Trash2 className="mr-2 h-4 w-4" />
                                            Delete
                                        </DropdownMenuItem>
                                    </DropdownMenuContent>
                                </DropdownMenu>
                            </div>
                        </div>
                    </div>
                </div>

                <div className="p-6 space-y-6">
                    {/* Hero Section */}
                    <div className="relative overflow-hidden rounded-2xl bg-gradient-to-br from-primary/5 via-primary/10 to-transparent border">
                        <div className="absolute inset-0 bg-grid-white/5" />
                        <div className="relative p-6 md:p-8">
                            <div className="flex flex-col md:flex-row md:items-start md:justify-between gap-6">
                                <div className="space-y-4">
                                    <div className="flex items-center gap-3">
                                        <Badge
                                            className={cn(
                                                'px-3 py-1.5 font-medium rounded-full flex items-center gap-1.5',
                                                statusConfig.color
                                            )}
                                        >
                                            <StatusIcon className="h-3.5 w-3.5" />
                                            {statusConfig.label}
                                        </Badge>
                                        {property.on_the_market && (
                                            <Badge
                                                variant="outline"
                                                className="px-3 py-1.5 rounded-full bg-background/50"
                                            >
                                                <Globe className="mr-1.5 h-3.5 w-3.5" />
                                                On Market
                                            </Badge>
                                        )}
                                        {property.feature_on_homepage && (
                                            <Badge
                                                variant="outline"
                                                className="px-3 py-1.5 rounded-full bg-background/50"
                                            >
                                                <Sparkles className="mr-1.5 h-3.5 w-3.5 text-amber-500" />
                                                Featured
                                            </Badge>
                                        )}
                                    </div>

                                    <div>
                                        <h2 className="text-2xl md:text-3xl font-bold">
                                            {property.display_address || 'Property Address'}
                                        </h2>
                                        <div className="flex items-center gap-2 mt-2 text-muted-foreground">
                                            <MapPin className="h-4 w-4" />
                                            <p>
                                                {[property.town_city, property.county, property.postcode]
                                                    .filter(Boolean)
                                                    .join(', ') || 'Location not specified'}
                                            </p>
                                        </div>
                                    </div>

                                    <div className="flex flex-wrap items-center gap-6 pt-2">
                                        {property.number_of_bedrooms !== null && (
                                            <div className="flex items-center gap-2">
                                                <div className="p-2 rounded-lg bg-background/80">
                                                    <Bed className="h-4 w-4 text-muted-foreground" />
                                                </div>
                                                <div>
                                                    <p className="text-lg font-semibold">{property.number_of_bedrooms}</p>
                                                    <p className="text-xs text-muted-foreground">Beds</p>
                                                </div>
                                            </div>
                                        )}
                                        {property.number_of_bathrooms !== null && (
                                            <div className="flex items-center gap-2">
                                                <div className="p-2 rounded-lg bg-background/80">
                                                    <Bath className="h-4 w-4 text-muted-foreground" />
                                                </div>
                                                <div>
                                                    <p className="text-lg font-semibold">{property.number_of_bathrooms}</p>
                                                    <p className="text-xs text-muted-foreground">Baths</p>
                                                </div>
                                            </div>
                                        )}
                                        {property.number_of_receptions !== null && (
                                            <div className="flex items-center gap-2">
                                                <div className="p-2 rounded-lg bg-background/80">
                                                    <Sofa className="h-4 w-4 text-muted-foreground" />
                                                </div>
                                                <div>
                                                    <p className="text-lg font-semibold">{property.number_of_receptions}</p>
                                                    <p className="text-xs text-muted-foreground">Receptions</p>
                                                </div>
                                            </div>
                                        )}
                                        {property.property_type && (
                                            <div className="flex items-center gap-2">
                                                <div className="p-2 rounded-lg bg-background/80">
                                                    <Home className="h-4 w-4 text-muted-foreground" />
                                                </div>
                                                <div>
                                                    <p className="text-lg font-semibold">{property.property_type}</p>
                                                    <p className="text-xs text-muted-foreground">Type</p>
                                                </div>
                                            </div>
                                        )}
                                    </div>
                                </div>

                                <div className="flex flex-col items-end gap-4">
                                    <div className="text-right">
                                        <p className="text-sm text-muted-foreground">
                                            {property.price_qualifier || 'Price'}
                                        </p>
                                        <p className="text-3xl md:text-4xl font-bold text-primary">
                                            {property.price ? `£${property.price.toLocaleString()}` : 'TBC'}
                                        </p>
                                        {property.business_category === 'lettings' && (
                                            <p className="text-sm text-muted-foreground">per month</p>
                                        )}
                                    </div>

                                    {property.landlord && (
                                        <Link
                                            to={`/contacts/${property.landlord.id}`}
                                            className="flex items-center gap-3 p-3 rounded-xl bg-background/80 hover:bg-background transition-colors"
                                        >
                                            <Avatar className="h-10 w-10">
                                                <AvatarFallback className="bg-primary/10 text-primary text-sm">
                                                    {property.landlord.first_name?.[0]}
                                                    {property.landlord.last_name?.[0]}
                                                </AvatarFallback>
                                            </Avatar>
                                            <div className="text-left">
                                                <p className="text-sm font-medium">
                                                    {property.landlord.first_name} {property.landlord.last_name}
                                                </p>
                                                <p className="text-xs text-muted-foreground">Landlord</p>
                                            </div>
                                            <ChevronRight className="h-4 w-4 text-muted-foreground" />
                                        </Link>
                                    )}
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Quick Stats */}
                    <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                        <Card className="border-0 shadow-sm bg-gradient-to-br from-emerald-500/5 to-transparent">
                            <CardContent className="p-4">
                                <div className="flex items-center gap-3">
                                    <div className="p-2.5 rounded-xl bg-emerald-500/10">
                                        <PoundSterling className="h-5 w-5 text-emerald-600" />
                                    </div>
                                    <div>
                                        <p className="text-xs text-muted-foreground font-medium">Valuation</p>
                                        <p className="text-lg font-bold">
                                            {property.valuation_price
                                                ? `£${property.valuation_price.toLocaleString()}`
                                                : '—'}
                                        </p>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>

                        <Card className="border-0 shadow-sm bg-gradient-to-br from-blue-500/5 to-transparent">
                            <CardContent className="p-4">
                                <div className="flex items-center gap-3">
                                    <div className="p-2.5 rounded-xl bg-blue-500/10">
                                        <Gauge className="h-5 w-5 text-blue-600" />
                                    </div>
                                    <div>
                                        <p className="text-xs text-muted-foreground font-medium">EPC Rating</p>
                                        <p className="text-lg font-bold">
                                            {property.epc_band || property.epc_rating || '—'}
                                        </p>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>

                        <Card className="border-0 shadow-sm bg-gradient-to-br from-purple-500/5 to-transparent">
                            <CardContent className="p-4">
                                <div className="flex items-center gap-3">
                                    <div className="p-2.5 rounded-xl bg-purple-500/10">
                                        <Calendar className="h-5 w-5 text-purple-600" />
                                    </div>
                                    <div>
                                        <p className="text-xs text-muted-foreground font-medium">Available</p>
                                        <p className="text-lg font-bold">
                                            {property.date_available
                                                ? format(new Date(property.date_available), 'dd MMM')
                                                : '—'}
                                        </p>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>

                        <Card className="border-0 shadow-sm bg-gradient-to-br from-amber-500/5 to-transparent">
                            <CardContent className="p-4">
                                <div className="flex items-center gap-3">
                                    <div className="p-2.5 rounded-xl bg-amber-500/10">
                                        <Shield className="h-5 w-5 text-amber-600" />
                                    </div>
                                    <div>
                                        <p className="text-xs text-muted-foreground font-medium">Deposit</p>
                                        <p className="text-lg font-bold">
                                            {property.deposit ? `£${property.deposit.toLocaleString()}` : '—'}
                                        </p>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>
                    </div>

                    {/* Tabbed Content */}
                    <Tabs defaultValue="details" className="space-y-6">
                        <div className="border-b">
                            <ScrollArea className="w-full">
                                <TabsList className="inline-flex h-12 items-center justify-start gap-1 bg-transparent p-0">
                                    <TabsTrigger
                                        value="details"
                                        className="inline-flex items-center gap-2 rounded-none border-b-2 border-transparent px-4 py-3 text-sm font-medium text-muted-foreground transition-colors hover:text-foreground data-[state=active]:border-primary data-[state=active]:text-foreground data-[state=active]:bg-transparent"
                                    >
                                        <Building2 className="h-4 w-4" />
                                        Details
                                    </TabsTrigger>
                                    <TabsTrigger
                                        value="description"
                                        className="inline-flex items-center gap-2 rounded-none border-b-2 border-transparent px-4 py-3 text-sm font-medium text-muted-foreground transition-colors hover:text-foreground data-[state=active]:border-primary data-[state=active]:text-foreground data-[state=active]:bg-transparent"
                                    >
                                        <FileText className="h-4 w-4" />
                                        Description
                                    </TabsTrigger>
                                    <TabsTrigger
                                        value="financials"
                                        className="inline-flex items-center gap-2 rounded-none border-b-2 border-transparent px-4 py-3 text-sm font-medium text-muted-foreground transition-colors hover:text-foreground data-[state=active]:border-primary data-[state=active]:text-foreground data-[state=active]:bg-transparent"
                                    >
                                        <PoundSterling className="h-4 w-4" />
                                        Financials
                                    </TabsTrigger>
                                    <TabsTrigger
                                        value="compliance"
                                        className="inline-flex items-center gap-2 rounded-none border-b-2 border-transparent px-4 py-3 text-sm font-medium text-muted-foreground transition-colors hover:text-foreground data-[state=active]:border-primary data-[state=active]:text-foreground data-[state=active]:bg-transparent"
                                    >
                                        <Shield className="h-4 w-4" />
                                        Compliance
                                    </TabsTrigger>
                                    <TabsTrigger
                                        value="tenancies"
                                        className="inline-flex items-center gap-2 rounded-none border-b-2 border-transparent px-4 py-3 text-sm font-medium text-muted-foreground transition-colors hover:text-foreground data-[state=active]:border-primary data-[state=active]:text-foreground data-[state=active]:bg-transparent"
                                    >
                                        <Key className="h-4 w-4" />
                                        Tenancies
                                    </TabsTrigger>
                                    <TabsTrigger
                                        value="notes"
                                        className="inline-flex items-center gap-2 rounded-none border-b-2 border-transparent px-4 py-3 text-sm font-medium text-muted-foreground transition-colors hover:text-foreground data-[state=active]:border-primary data-[state=active]:text-foreground data-[state=active]:bg-transparent"
                                    >
                                        <MessageSquare className="h-4 w-4" />
                                        Notes
                                    </TabsTrigger>
                                    <TabsTrigger
                                        value="images"
                                        className="inline-flex items-center gap-2 rounded-none border-b-2 border-transparent px-4 py-3 text-sm font-medium text-muted-foreground transition-colors hover:text-foreground data-[state=active]:border-primary data-[state=active]:text-foreground data-[state=active]:bg-transparent"
                                    >
                                        <Image className="h-4 w-4" />
                                        Images
                                    </TabsTrigger>
                                    <TabsTrigger
                                        value="documents"
                                        className="inline-flex items-center gap-2 rounded-none border-b-2 border-transparent px-4 py-3 text-sm font-medium text-muted-foreground transition-colors hover:text-foreground data-[state=active]:border-primary data-[state=active]:text-foreground data-[state=active]:bg-transparent"
                                    >
                                        <FileCheck className="h-4 w-4" />
                                        Documents
                                    </TabsTrigger>
                                    <TabsTrigger
                                        value="activities"
                                        className="inline-flex items-center gap-2 rounded-none border-b-2 border-transparent px-4 py-3 text-sm font-medium text-muted-foreground transition-colors hover:text-foreground data-[state=active]:border-primary data-[state=active]:text-foreground data-[state=active]:bg-transparent"
                                    >
                                        <Activity className="h-4 w-4" />
                                        Activity
                                    </TabsTrigger>
                                </TabsList>
                                <ScrollBar orientation="horizontal" className="invisible" />
                            </ScrollArea>
                        </div>

                        <TabsContent value="details" className="space-y-6 mt-6">
                            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                                {/* Property Information */}
                                <Card className="border-0 shadow-sm">
                                    <CardHeader className="pb-4">
                                        <CardTitle className="text-base font-semibold flex items-center gap-2">
                                            <Building2 className="h-4 w-4 text-primary" />
                                            Property Information
                                        </CardTitle>
                                    </CardHeader>
                                    <CardContent className="grid grid-cols-2 gap-x-8 gap-y-4">
                                        <DetailItem label="Reference" value={property.reference} />
                                        <DetailItem label="Property Class" value={property.property_class} />
                                        <DetailItem label="Business Category" value={property.business_category} />
                                        <DetailItem label="Property Type" value={property.property_type} />
                                        <DetailItem label="Bedrooms" value={property.number_of_bedrooms} />
                                        <DetailItem label="Bathrooms" value={property.number_of_bathrooms} />
                                        <DetailItem label="Receptions" value={property.number_of_receptions} />
                                        <DetailItem label="Council Tax Band" value={property.council_tax_band} />
                                        <DetailItem label="Local Authority" value={property.local_authority} />
                                        <DetailItem
                                            label="Date Available"
                                            value={
                                                property.date_available
                                                    ? format(new Date(property.date_available), 'dd MMM yyyy')
                                                    : null
                                            }
                                        />
                                    </CardContent>
                                </Card>

                                {/* Address */}
                                <Card className="border-0 shadow-sm">
                                    <CardHeader className="pb-4">
                                        <CardTitle className="text-base font-semibold flex items-center gap-2">
                                            <MapPin className="h-4 w-4 text-primary" />
                                            Address
                                        </CardTitle>
                                    </CardHeader>
                                    <CardContent className="grid grid-cols-2 gap-x-8 gap-y-4">
                                        <DetailItem
                                            label="Display Address"
                                            value={property.display_address}
                                            className="col-span-2"
                                        />
                                        <DetailItem label="Address Line 1" value={property.address_line_1} />
                                        <DetailItem label="Address Line 2" value={property.address_line_2} />
                                        <DetailItem label="Town / City" value={property.town_city} />
                                        <DetailItem label="County" value={property.county} />
                                        <DetailItem label="Postcode" value={property.postcode} />
                                        <DetailItem label="Country" value={property.country} />
                                        <DetailItem
                                            label="Is Overseas"
                                            value={property.is_overseas ? 'Yes' : 'No'}
                                        />
                                    </CardContent>
                                </Card>

                                {/* Management */}
                                <Card className="border-0 shadow-sm">
                                    <CardHeader className="pb-4">
                                        <CardTitle className="text-base font-semibold flex items-center gap-2">
                                            <User className="h-4 w-4 text-primary" />
                                            Management
                                        </CardTitle>
                                    </CardHeader>
                                    <CardContent className="grid grid-cols-2 gap-x-8 gap-y-4">
                                        <DetailItem label="Branch" value={property.branch} />
                                        <DetailItem label="Lead Staff" value={property.lead_staff} />
                                        <DetailItem label="Marketing Tag" value={property.marketing_tag} />
                                        <DetailItem
                                            label="On The Market"
                                            value={property.on_the_market ? 'Yes' : 'No'}
                                        />
                                        <DetailItem
                                            label="Put On Market"
                                            value={
                                                property.put_on_market
                                                    ? format(new Date(property.put_on_market), 'dd MMM yyyy')
                                                    : null
                                            }
                                        />
                                    </CardContent>
                                </Card>

                                {/* Portal Settings */}
                                <Card className="border-0 shadow-sm">
                                    <CardHeader className="pb-4">
                                        <CardTitle className="text-base font-semibold flex items-center gap-2">
                                            <Globe className="h-4 w-4 text-primary" />
                                            Portal Settings
                                        </CardTitle>
                                    </CardHeader>
                                    <CardContent className="space-y-4">
                                        <div className="flex items-center justify-between py-2">
                                            <span className="text-sm text-muted-foreground">
                                                Exclude from All Portals
                                            </span>
                                            <Badge
                                                variant={property.exclude_from_all_portals ? 'destructive' : 'secondary'}
                                                className="text-xs"
                                            >
                                                {property.exclude_from_all_portals ? 'Yes' : 'No'}
                                            </Badge>
                                        </div>
                                        <Separator />
                                        <div className="flex items-center justify-between py-2">
                                            <span className="text-sm text-muted-foreground">
                                                Exclude from Website
                                            </span>
                                            <Badge
                                                variant={property.exclude_from_website ? 'destructive' : 'secondary'}
                                                className="text-xs"
                                            >
                                                {property.exclude_from_website ? 'Yes' : 'No'}
                                            </Badge>
                                        </div>
                                        <Separator />
                                        <div className="flex items-center justify-between py-2">
                                            <span className="text-sm text-muted-foreground">
                                                Exclude from Window Rotator
                                            </span>
                                            <Badge
                                                variant={property.exclude_from_window_rotator ? 'destructive' : 'secondary'}
                                                className="text-xs"
                                            >
                                                {property.exclude_from_window_rotator ? 'Yes' : 'No'}
                                            </Badge>
                                        </div>
                                        <Separator />
                                        <div className="flex items-center justify-between py-2">
                                            <span className="text-sm text-muted-foreground">Feature on Homepage</span>
                                            <Badge
                                                variant={property.feature_on_homepage ? 'default' : 'secondary'}
                                                className="text-xs"
                                            >
                                                {property.feature_on_homepage ? 'Yes' : 'No'}
                                            </Badge>
                                        </div>
                                    </CardContent>
                                </Card>
                            </div>
                        </TabsContent>

                        <TabsContent value="description" className="space-y-6 mt-6">
                            <Card className="border-0 shadow-sm">
                                <CardHeader>
                                    <CardTitle className="text-base font-semibold">Description</CardTitle>
                                </CardHeader>
                                <CardContent className="space-y-8">
                                    <div className="space-y-2">
                                        <h4 className="text-sm font-medium text-muted-foreground uppercase tracking-wider">
                                            Headline
                                        </h4>
                                        <p className="text-lg font-medium">
                                            {property.headline || (
                                                <span className="text-muted-foreground italic">No headline</span>
                                            )}
                                        </p>
                                    </div>

                                    <Separator />

                                    <div className="space-y-2">
                                        <h4 className="text-sm font-medium text-muted-foreground uppercase tracking-wider">
                                            Summary
                                        </h4>
                                        <p className="text-muted-foreground leading-relaxed">
                                            {property.summary || (
                                                <span className="italic">No summary</span>
                                            )}
                                        </p>
                                    </div>

                                    <Separator />

                                    <div className="space-y-2">
                                        <h4 className="text-sm font-medium text-muted-foreground uppercase tracking-wider">
                                            Full Description
                                        </h4>
                                        <div className="prose prose-sm max-w-none text-muted-foreground">
                                            <p className="whitespace-pre-wrap leading-relaxed">
                                                {property.full_description || (
                                                    <span className="italic">No description</span>
                                                )}
                                            </p>
                                        </div>
                                    </div>

                                    <Separator />

                                    <div className="space-y-2">
                                        <h4 className="text-sm font-medium text-muted-foreground uppercase tracking-wider">
                                            Viewing Arrangements
                                        </h4>
                                        <p className="text-muted-foreground">
                                            {property.viewing_arrangements || (
                                                <span className="italic">No viewing arrangements</span>
                                            )}
                                        </p>
                                    </div>

                                    <Separator />

                                    <div className="space-y-2">
                                        <h4 className="text-sm font-medium text-muted-foreground uppercase tracking-wider">
                                            Additional Information
                                        </h4>
                                        <p className="text-muted-foreground">
                                            {property.additional_information || (
                                                <span className="italic">No additional information</span>
                                            )}
                                        </p>
                                    </div>

                                    {property.dashboard_note && (
                                        <>
                                            <Separator />
                                            <div className="p-4 rounded-xl bg-amber-500/5 border border-amber-500/20">
                                                <h4 className="text-sm font-medium text-amber-600 uppercase tracking-wider mb-2 flex items-center gap-2">
                                                    <AlertCircle className="h-4 w-4" />
                                                    Dashboard Note
                                                </h4>
                                                <p className="text-amber-700">{property.dashboard_note}</p>
                                            </div>
                                        </>
                                    )}
                                </CardContent>
                            </Card>

                            {/* External Links */}
                            <Card className="border-0 shadow-sm">
                                <CardHeader>
                                    <CardTitle className="text-base font-semibold flex items-center gap-2">
                                        <Link2 className="h-4 w-4 text-primary" />
                                        External Links
                                    </CardTitle>
                                </CardHeader>
                                <CardContent>
                                    <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                                        {property.link_on_website && (
                                            <a
                                                href={property.link_on_website}
                                                target="_blank"
                                                rel="noopener noreferrer"
                                                className="flex items-center gap-3 p-4 rounded-xl bg-muted/50 hover:bg-muted transition-colors group"
                                            >
                                                <div className="p-2 rounded-lg bg-background">
                                                    <Globe className="h-4 w-4 text-muted-foreground group-hover:text-primary transition-colors" />
                                                </div>
                                                <div>
                                                    <p className="font-medium text-sm">Website</p>
                                                    <p className="text-xs text-muted-foreground">View listing</p>
                                                </div>
                                                <ExternalLink className="h-4 w-4 ml-auto text-muted-foreground" />
                                            </a>
                                        )}
                                        {property.rightmove_link && (
                                            <a
                                                href={property.rightmove_link}
                                                target="_blank"
                                                rel="noopener noreferrer"
                                                className="flex items-center gap-3 p-4 rounded-xl bg-muted/50 hover:bg-muted transition-colors group"
                                            >
                                                <div className="p-2 rounded-lg bg-background">
                                                    <Home className="h-4 w-4 text-muted-foreground group-hover:text-primary transition-colors" />
                                                </div>
                                                <div>
                                                    <p className="font-medium text-sm">Rightmove</p>
                                                    <p className="text-xs text-muted-foreground">View listing</p>
                                                </div>
                                                <ExternalLink className="h-4 w-4 ml-auto text-muted-foreground" />
                                            </a>
                                        )}
                                        {property.zoopla_link && (
                                            <a
                                                href={property.zoopla_link}
                                                target="_blank"
                                                rel="noopener noreferrer"
                                                className="flex items-center gap-3 p-4 rounded-xl bg-muted/50 hover:bg-muted transition-colors group"
                                            >
                                                <div className="p-2 rounded-lg bg-background">
                                                    <Home className="h-4 w-4 text-muted-foreground group-hover:text-primary transition-colors" />
                                                </div>
                                                <div>
                                                    <p className="font-medium text-sm">Zoopla</p>
                                                    <p className="text-xs text-muted-foreground">View listing</p>
                                                </div>
                                                <ExternalLink className="h-4 w-4 ml-auto text-muted-foreground" />
                                            </a>
                                        )}
                                        {property.on_the_market_link && (
                                            <a
                                                href={property.on_the_market_link}
                                                target="_blank"
                                                rel="noopener noreferrer"
                                                className="flex items-center gap-3 p-4 rounded-xl bg-muted/50 hover:bg-muted transition-colors group"
                                            >
                                                <div className="p-2 rounded-lg bg-background">
                                                    <Home className="h-4 w-4 text-muted-foreground group-hover:text-primary transition-colors" />
                                                </div>
                                                <div>
                                                    <p className="font-medium text-sm">OnTheMarket</p>
                                                    <p className="text-xs text-muted-foreground">View listing</p>
                                                </div>
                                                <ExternalLink className="h-4 w-4 ml-auto text-muted-foreground" />
                                            </a>
                                        )}
                                        {property.virtual_tour_link && (
                                            <a
                                                href={property.virtual_tour_link}
                                                target="_blank"
                                                rel="noopener noreferrer"
                                                className="flex items-center gap-3 p-4 rounded-xl bg-muted/50 hover:bg-muted transition-colors group"
                                            >
                                                <div className="p-2 rounded-lg bg-background">
                                                    <Image className="h-4 w-4 text-muted-foreground group-hover:text-primary transition-colors" />
                                                </div>
                                                <div>
                                                    <p className="font-medium text-sm">Virtual Tour</p>
                                                    <p className="text-xs text-muted-foreground">360° view</p>
                                                </div>
                                                <ExternalLink className="h-4 w-4 ml-auto text-muted-foreground" />
                                            </a>
                                        )}
                                        {property.virtual_tour_link_2 && (
                                            <a
                                                href={property.virtual_tour_link_2}
                                                target="_blank"
                                                rel="noopener noreferrer"
                                                className="flex items-center gap-3 p-4 rounded-xl bg-muted/50 hover:bg-muted transition-colors group"
                                            >
                                                <div className="p-2 rounded-lg bg-background">
                                                    <Image className="h-4 w-4 text-muted-foreground group-hover:text-primary transition-colors" />
                                                </div>
                                                <div>
                                                    <p className="font-medium text-sm">Virtual Tour 2</p>
                                                    <p className="text-xs text-muted-foreground">360° view</p>
                                                </div>
                                                <ExternalLink className="h-4 w-4 ml-auto text-muted-foreground" />
                                            </a>
                                        )}
                                    </div>
                                    {!property.link_on_website &&
                                        !property.rightmove_link &&
                                        !property.zoopla_link &&
                                        !property.on_the_market_link &&
                                        !property.virtual_tour_link &&
                                        !property.virtual_tour_link_2 && (
                                            <div className="text-center py-8 text-muted-foreground">
                                                <Link2 className="h-8 w-8 mx-auto mb-2 opacity-50" />
                                                <p>No external links configured</p>
                                            </div>
                                        )}
                                </CardContent>
                            </Card>
                        </TabsContent>

                        <TabsContent value="financials" className="space-y-6 mt-6">
                            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                                {/* Pricing */}
                                <Card className="border-0 shadow-sm">
                                    <CardHeader className="pb-4">
                                        <CardTitle className="text-base font-semibold flex items-center gap-2">
                                            <PoundSterling className="h-4 w-4 text-primary" />
                                            Pricing
                                        </CardTitle>
                                    </CardHeader>
                                    <CardContent className="space-y-4">
                                        <div className="p-4 rounded-xl bg-primary/5 border border-primary/10">
                                            <p className="text-sm text-muted-foreground mb-1">
                                                {property.price_qualifier || 'Asking Price'}
                                            </p>
                                            <p className="text-3xl font-bold text-primary">
                                                {property.price ? `£${property.price.toLocaleString()}` : 'TBC'}
                                            </p>
                                        </div>

                                        <div className="grid grid-cols-2 gap-4">
                                            <div className="p-4 rounded-xl bg-muted/50">
                                                <p className="text-xs text-muted-foreground mb-1">Valuation</p>
                                                <p className="text-lg font-semibold">
                                                    {property.valuation_price
                                                        ? `£${property.valuation_price.toLocaleString()}`
                                                        : '—'}
                                                </p>
                                            </div>
                                            <div className="p-4 rounded-xl bg-muted/50">
                                                <p className="text-xs text-muted-foreground mb-1">Minimum Acceptable</p>
                                                <p className="text-lg font-semibold">
                                                    {property.minimum_acceptable_price
                                                        ? `£${property.minimum_acceptable_price.toLocaleString()}`
                                                        : '—'}
                                                </p>
                                            </div>
                                        </div>
                                    </CardContent>
                                </Card>

                                {/* Deposits & Charges */}
                                <Card className="border-0 shadow-sm">
                                    <CardHeader className="pb-4">
                                        <CardTitle className="text-base font-semibold flex items-center gap-2">
                                            <Shield className="h-4 w-4 text-primary" />
                                            Deposits & Charges
                                        </CardTitle>
                                    </CardHeader>
                                    <CardContent className="grid grid-cols-2 gap-x-8 gap-y-4">
                                        <DetailItem
                                            label="Deposit"
                                            value={
                                                property.deposit ? `£${property.deposit.toLocaleString()}` : null
                                            }
                                        />
                                        <DetailItem
                                            label="Ground Rent"
                                            value={
                                                property.ground_rent ? `£${property.ground_rent.toLocaleString()}` : null
                                            }
                                        />
                                        <DetailItem
                                            label="Maintenance Charge"
                                            value={
                                                property.maintenance_charge
                                                    ? `£${property.maintenance_charge.toLocaleString()}`
                                                    : null
                                            }
                                        />
                                    </CardContent>
                                </Card>

                                {/* Letting Fees */}
                                <Card className="border-0 shadow-sm">
                                    <CardHeader className="pb-4">
                                        <CardTitle className="text-base font-semibold flex items-center gap-2">
                                            <TrendingUp className="h-4 w-4 text-primary" />
                                            Letting Fees
                                        </CardTitle>
                                    </CardHeader>
                                    <CardContent className="grid grid-cols-2 gap-x-8 gap-y-4">
                                        <DetailItem label="Letting Service" value={property.letting_service} />
                                        <DetailItem
                                            label="Letting Fee"
                                            value={
                                                property.letting_fee ? `£${property.letting_fee.toLocaleString()}` : null
                                            }
                                        />
                                        <DetailItem
                                            label="Management Fee"
                                            value={
                                                property.management_fee
                                                    ? `£${property.management_fee.toLocaleString()}`
                                                    : null
                                            }
                                        />
                                        <DetailItem
                                            label="Payment Frequency"
                                            value={property.landlord_payment_frequency}
                                        />
                                    </CardContent>
                                </Card>

                                {/* Landlord */}
                                <Card className="border-0 shadow-sm">
                                    <CardHeader className="pb-4">
                                        <CardTitle className="text-base font-semibold flex items-center gap-2">
                                            <User className="h-4 w-4 text-primary" />
                                            Landlord
                                        </CardTitle>
                                    </CardHeader>
                                    <CardContent>
                                        {property.landlord ? (
                                            <Link
                                                to={`/contacts/${property.landlord.id}`}
                                                className="flex items-center gap-4 p-4 rounded-xl bg-muted/50 hover:bg-muted transition-colors group"
                                            >
                                                <Avatar className="h-12 w-12">
                                                    <AvatarFallback className="bg-primary/10 text-primary">
                                                        {property.landlord.first_name?.[0]}
                                                        {property.landlord.last_name?.[0]}
                                                    </AvatarFallback>
                                                </Avatar>
                                                <div className="flex-1">
                                                    <p className="font-semibold group-hover:text-primary transition-colors">
                                                        {property.landlord.first_name} {property.landlord.last_name}
                                                    </p>
                                                    {property.landlord.email && (
                                                        <p className="text-sm text-muted-foreground">
                                                            {property.landlord.email}
                                                        </p>
                                                    )}
                                                    {property.landlord.phone && (
                                                        <p className="text-sm text-muted-foreground">
                                                            {property.landlord.phone}
                                                        </p>
                                                    )}
                                                    {property.landlord.company && (
                                                        <Badge variant="secondary" className="mt-2 text-xs">
                                                            {property.landlord.company}
                                                        </Badge>
                                                    )}
                                                </div>
                                                <ChevronRight className="h-5 w-5 text-muted-foreground group-hover:text-primary transition-colors" />
                                            </Link>
                                        ) : (
                                            <div className="text-center py-8 text-muted-foreground">
                                                <User className="h-8 w-8 mx-auto mb-2 opacity-50" />
                                                <p>No landlord assigned</p>
                                            </div>
                                        )}
                                    </CardContent>
                                </Card>
                            </div>
                        </TabsContent>

                        <TabsContent value="compliance" className="space-y-6 mt-6">
                            {/* Compliance Overview */}
                            <Card className="border-0 shadow-sm overflow-hidden">
                                <CardHeader className="bg-gradient-to-r from-amber-500/5 to-transparent">
                                    <CardTitle className="text-base font-semibold flex items-center gap-2">
                                        <AlertTriangle className="h-4 w-4 text-amber-500" />
                                        Compliance & Certificates
                                    </CardTitle>
                                    <CardDescription>
                                        Track important compliance dates and certifications
                                    </CardDescription>
                                </CardHeader>
                                <CardContent className="pt-6">
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <ComplianceItem
                                            label="EPC Certificate"
                                            date={property.epc_expiry}
                                            icon={Gauge}
                                        />
                                        <ComplianceItem
                                            label="Gas Safety Certificate"
                                            date={property.gas_certificate_expiry}
                                            icon={Flame}
                                        />
                                        <ComplianceItem
                                            label="Electrical Certificate"
                                            date={property.electricity_certificate_expiry}
                                            icon={Zap}
                                        />
                                        <ComplianceItem
                                            label="PAT Test"
                                            date={property.pat_test_expiry}
                                            icon={FileCheck}
                                        />
                                        <ComplianceItem
                                            label="Insurance"
                                            date={property.insurance_expiry}
                                            icon={Shield}
                                        />
                                        <ComplianceItem
                                            label="Smoke & CO Alarm"
                                            date={property.smoke_co_alarm_expiry}
                                            icon={AlertCircle}
                                        />
                                        <ComplianceItem
                                            label="Property Licence"
                                            date={property.property_licence_expiry}
                                            icon={FileText}
                                        />
                                        <ComplianceItem
                                            label="Legionella Assessment"
                                            date={property.legionella_risk_assessment_date}
                                            icon={Activity}
                                        />
                                    </div>
                                </CardContent>
                            </Card>

                            {/* EPC Details */}
                            <Card className="border-0 shadow-sm">
                                <CardHeader className="pb-4">
                                    <CardTitle className="text-base font-semibold flex items-center gap-2">
                                        <Gauge className="h-4 w-4 text-primary" />
                                        EPC Details
                                    </CardTitle>
                                </CardHeader>
                                <CardContent className="grid grid-cols-3 gap-x-8 gap-y-4">
                                    <DetailItem label="EPC Rating" value={property.epc_rating} />
                                    <DetailItem label="EPC Band" value={property.epc_band} />
                                    <DetailItem
                                        label="EPC Expiry"
                                        value={
                                            property.epc_expiry
                                                ? format(new Date(property.epc_expiry), 'dd MMM yyyy')
                                                : null
                                        }
                                    />
                                </CardContent>
                            </Card>

                            {/* Gas & Utilities */}
                            <Card className="border-0 shadow-sm">
                                <CardHeader className="pb-4">
                                    <CardTitle className="text-base font-semibold flex items-center gap-2">
                                        <Flame className="h-4 w-4 text-primary" />
                                        Utilities & Safety
                                    </CardTitle>
                                </CardHeader>
                                <CardContent className="grid grid-cols-2 gap-x-8 gap-y-4">
                                    <DetailItem
                                        label="Has Gas"
                                        value={property.has_gas ? 'Yes' : 'No'}
                                    />
                                    <DetailItem
                                        label="Property Licence Type"
                                        value={property.property_licence_type}
                                    />
                                    <DetailItem
                                        label="Property Licence Notes"
                                        value={property.property_licence_notes}
                                        className="col-span-2"
                                    />
                                </CardContent>
                            </Card>

                            {/* Additional Fields */}
                            {(property.additional_field_1 || property.additional_field_2) && (
                                <Card className="border-0 shadow-sm">
                                    <CardHeader className="pb-4">
                                        <CardTitle className="text-base font-semibold">Additional Fields</CardTitle>
                                    </CardHeader>
                                    <CardContent className="grid grid-cols-2 gap-x-8 gap-y-4">
                                        <DetailItem label="Additional Field 1" value={property.additional_field_1} />
                                        <DetailItem label="Additional Field 2" value={property.additional_field_2} />
                                    </CardContent>
                                </Card>
                            )}
                        </TabsContent>

                        <TabsContent value="tenancies" className="mt-6">
                            <PropertyTenanciesTab propertyId={id!} />
                        </TabsContent>

                        <TabsContent value="notes" className="mt-6">
                            <PropertyNotesTab propertyId={id!} />
                        </TabsContent>

                        <TabsContent value="images" className="mt-6">
                            <PropertyImagesTab propertyId={id!} />
                        </TabsContent>

                        <TabsContent value="documents" className="mt-6">
                            <PropertyDocumentsTab propertyId={id!} />
                        </TabsContent>

                        <TabsContent value="activities" className="mt-6">
                            <PropertyActivitiesTab propertyId={id!} />
                        </TabsContent>
                    </Tabs>
                </div>
            </div>

            {/* Edit Dialog */}
            <PropertyFormDialog open={editOpen} onOpenChange={setEditOpen} propertyId={id} />

            {/* Delete Dialog */}
            <AlertDialog open={deleteOpen} onOpenChange={setDeleteOpen}>
                <AlertDialogContent className="sm:max-w-md">
                    <AlertDialogHeader>
                        <div className="w-12 h-12 rounded-full bg-destructive/10 flex items-center justify-center mx-auto mb-4">
                            <Trash2 className="h-6 w-6 text-destructive" />
                        </div>
                        <AlertDialogTitle className="text-center">Delete Property</AlertDialogTitle>
                        <AlertDialogDescription className="text-center">
                            Are you sure you want to delete this property? This action can be undone from the
                            trash.
                        </AlertDialogDescription>
                    </AlertDialogHeader>
                    <AlertDialogFooter className="sm:justify-center gap-2 mt-4">
                        <AlertDialogCancel className="rounded-xl">Cancel</AlertDialogCancel>
                        <AlertDialogAction
                            onClick={handleDelete}
                            className="bg-destructive text-destructive-foreground hover:bg-destructive/90 rounded-xl"
                        >
                            Delete Property
                        </AlertDialogAction>
                    </AlertDialogFooter>
                </AlertDialogContent>
            </AlertDialog>
        </div>
    );
}