import { motion } from 'framer-motion';
import { mockContacts, mockPipelines } from '@/lib/mock-data';
import { Contact } from '@/lib/types';
import { Avatar, AvatarFallback } from '@/components/ui/avatar';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Plus, Settings2, GripVertical, Building2 } from 'lucide-react';
import { cn } from '@/lib/utils';

export default function Pipelines() {
  const pipeline = mockPipelines[0];

  const getContactsByStage = (stageId: string): Contact[] => {
    return mockContacts.filter((c) => c.stageId === stageId);
  };

  const getInitials = (firstName: string, lastName: string) => {
    return `${firstName[0]}${lastName[0]}`.toUpperCase();
  };

  return (
    <div className="p-6 space-y-6 animate-fade-in h-[calc(100vh-8rem)]">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold tracking-tight">Pipelines</h1>
          <p className="text-muted-foreground mt-1">
            Visualize and manage your sales process
          </p>
        </div>
        <div className="flex items-center gap-2">
          <Button variant="outline" className="gap-2">
            <Settings2 className="h-4 w-4" />
            Configure
          </Button>
          <Button className="gradient-primary shadow-glow gap-2">
            <Plus className="h-4 w-4" />
            Add Deal
          </Button>
        </div>
      </div>

      {/* Pipeline selector */}
      <div className="flex items-center gap-2">
        {mockPipelines.map((p) => (
          <Badge
            key={p.id}
            variant={p.id === pipeline.id ? 'default' : 'secondary'}
            className="cursor-pointer px-3 py-1"
          >
            {p.name}
            {p.isDefault && (
              <span className="ml-1 text-xs opacity-70">(Default)</span>
            )}
          </Badge>
        ))}
      </div>

      {/* Kanban board */}
      <div className="flex gap-4 overflow-x-auto pb-4 flex-1">
        {pipeline.stages.map((stage, stageIndex) => {
          const contacts = getContactsByStage(stage.id);
          return (
            <motion.div
              key={stage.id}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: stageIndex * 0.1 }}
              className="flex-shrink-0 w-80"
            >
              <div className="rounded-xl border border-border bg-card overflow-hidden">
                {/* Stage header */}
                <div
                  className="p-4 border-b border-border"
                  style={{
                    background: `linear-gradient(135deg, ${stage.color}10 0%, transparent 100%)`,
                  }}
                >
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <div
                        className="h-3 w-3 rounded-full"
                        style={{ backgroundColor: stage.color }}
                      />
                      <h3 className="font-semibold">{stage.name}</h3>
                      <Badge variant="secondary" className="text-xs">
                        {contacts.length}
                      </Badge>
                    </div>
                    {stage.probability !== undefined && (
                      <span className="text-xs text-muted-foreground">
                        {stage.probability}%
                      </span>
                    )}
                  </div>
                </div>

                {/* Cards */}
                <div className="p-2 space-y-2 min-h-[200px] max-h-[calc(100vh-20rem)] overflow-y-auto">
                  {contacts.map((contact, index) => (
                    <motion.div
                      key={contact.id}
                      initial={{ opacity: 0, scale: 0.95 }}
                      animate={{ opacity: 1, scale: 1 }}
                      transition={{ delay: stageIndex * 0.1 + index * 0.05 }}
                      className="group cursor-pointer rounded-lg border border-border bg-background p-3 transition-all hover:shadow-md hover:border-primary/30"
                    >
                      <div className="flex items-start gap-3">
                        <GripVertical className="h-4 w-4 text-muted-foreground opacity-0 group-hover:opacity-100 transition-opacity mt-0.5 cursor-grab" />
                        <Avatar className="h-8 w-8">
                          <AvatarFallback className="bg-primary/10 text-primary text-xs">
                            {getInitials(contact.firstName, contact.lastName)}
                          </AvatarFallback>
                        </Avatar>
                        <div className="flex-1 min-w-0">
                          <p className="font-medium truncate">
                            {contact.firstName} {contact.lastName}
                          </p>
                          {contact.company && (
                            <p className="text-sm text-muted-foreground truncate flex items-center gap-1 mt-0.5">
                              <Building2 className="h-3 w-3" />
                              {contact.company}
                            </p>
                          )}
                          {contact.tags.length > 0 && (
                            <div className="flex items-center gap-1 mt-2">
                              {contact.tags.slice(0, 2).map((tag) => (
                                <span
                                  key={tag.id}
                                  className="inline-flex h-5 items-center rounded-full px-2 text-[10px] font-medium"
                                  style={{
                                    backgroundColor: `${tag.color}20`,
                                    color: tag.color,
                                  }}
                                >
                                  {tag.name}
                                </span>
                              ))}
                            </div>
                          )}
                        </div>
                      </div>
                    </motion.div>
                  ))}

                  {contacts.length === 0 && (
                    <div className="text-center py-8 text-muted-foreground text-sm">
                      No contacts in this stage
                    </div>
                  )}
                </div>

                {/* Add button */}
                <div className="p-2 border-t border-border">
                  <Button
                    variant="ghost"
                    size="sm"
                    className="w-full justify-start gap-2 text-muted-foreground"
                  >
                    <Plus className="h-4 w-4" />
                    Add contact
                  </Button>
                </div>
              </div>
            </motion.div>
          );
        })}
      </div>
    </div>
  );
}
