import { useState } from 'react';
import { Filter, X, ChevronDown, ChevronUp } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import {
  Collapsible,
  CollapsibleContent,
  CollapsibleTrigger,
} from '@/components/ui/collapsible';
import { Badge } from '@/components/ui/badge';

interface PropertyFiltersProps {
  filters: {
    status: string;
    landlordId: string;
    minBedrooms: string;
    maxBedrooms: string;
    minPrice: string;
    maxPrice: string;
    complianceExpiring: string;
  };
  onFiltersChange: (filters: PropertyFiltersProps['filters']) => void;
  landlords: { id: string; first_name: string; last_name: string }[];
}

export function PropertyFilters({ filters, onFiltersChange, landlords }: PropertyFiltersProps) {
  const [isOpen, setIsOpen] = useState(false);

  const updateFilter = (key: keyof typeof filters, value: string) => {
    onFiltersChange({ ...filters, [key]: value });
  };

  const clearFilters = () => {
    onFiltersChange({
      status: 'all',
      landlordId: 'all',
      minBedrooms: '',
      maxBedrooms: '',
      minPrice: '',
      maxPrice: '',
      complianceExpiring: 'all',
    });
  };

  const activeFilterCount = Object.entries(filters).filter(([key, value]) => {
    if (key === 'status' || key === 'landlordId' || key === 'complianceExpiring') {
      return value !== 'all';
    }
    return value !== '';
  }).length;

  return (
    <Collapsible open={isOpen} onOpenChange={setIsOpen}>
      <div className="flex items-center justify-between">
        <CollapsibleTrigger asChild>
          <Button variant="outline" className="gap-2">
            <Filter className="h-4 w-4" />
            Advanced Filters
            {activeFilterCount > 0 && (
              <Badge variant="secondary" className="ml-1">
                {activeFilterCount}
              </Badge>
            )}
            {isOpen ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
          </Button>
        </CollapsibleTrigger>
        {activeFilterCount > 0 && (
          <Button variant="ghost" size="sm" onClick={clearFilters} className="gap-1">
            <X className="h-3 w-3" />
            Clear filters
          </Button>
        )}
      </div>

      <CollapsibleContent className="mt-4">
        <div className="grid gap-4 p-4 border rounded-lg bg-muted/20">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
            {/* Status */}
            <div className="space-y-2">
              <Label>Status</Label>
              <Select value={filters.status} onValueChange={(v) => updateFilter('status', v)}>
                <SelectTrigger>
                  <SelectValue placeholder="All statuses" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Statuses</SelectItem>
                  <SelectItem value="available">Available</SelectItem>
                  <SelectItem value="let">Let</SelectItem>
                  <SelectItem value="under_offer">Under Offer</SelectItem>
                  <SelectItem value="sold">Sold</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {/* Landlord */}
            <div className="space-y-2">
              <Label>Landlord</Label>
              <Select value={filters.landlordId} onValueChange={(v) => updateFilter('landlordId', v)}>
                <SelectTrigger>
                  <SelectValue placeholder="All landlords" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Landlords</SelectItem>
                  {landlords.map((landlord) => (
                    <SelectItem key={landlord.id} value={landlord.id}>
                      {landlord.first_name} {landlord.last_name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            {/* Bedrooms */}
            <div className="space-y-2">
              <Label>Bedrooms</Label>
              <div className="flex gap-2">
                <Input
                  type="number"
                  placeholder="Min"
                  value={filters.minBedrooms}
                  onChange={(e) => updateFilter('minBedrooms', e.target.value)}
                  min={0}
                />
                <Input
                  type="number"
                  placeholder="Max"
                  value={filters.maxBedrooms}
                  onChange={(e) => updateFilter('maxBedrooms', e.target.value)}
                  min={0}
                />
              </div>
            </div>

            {/* Price */}
            <div className="space-y-2">
              <Label>Price (£)</Label>
              <div className="flex gap-2">
                <Input
                  type="number"
                  placeholder="Min"
                  value={filters.minPrice}
                  onChange={(e) => updateFilter('minPrice', e.target.value)}
                  min={0}
                />
                <Input
                  type="number"
                  placeholder="Max"
                  value={filters.maxPrice}
                  onChange={(e) => updateFilter('maxPrice', e.target.value)}
                  min={0}
                />
              </div>
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
            {/* Compliance Expiring */}
            <div className="space-y-2">
              <Label>Compliance Expiring</Label>
              <Select value={filters.complianceExpiring} onValueChange={(v) => updateFilter('complianceExpiring', v)}>
                <SelectTrigger>
                  <SelectValue placeholder="All" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Properties</SelectItem>
                  <SelectItem value="7">Within 7 days</SelectItem>
                  <SelectItem value="30">Within 30 days</SelectItem>
                  <SelectItem value="60">Within 60 days</SelectItem>
                  <SelectItem value="90">Within 90 days</SelectItem>
                  <SelectItem value="expired">Already Expired</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        </div>
      </CollapsibleContent>
    </Collapsible>
  );
}
