import { format } from 'date-fns';
import { 
  Mail, Phone, MessageSquare, Settings, Bot, Loader2 
} from 'lucide-react';
import { usePropertyActivities, PropertyActivity, ActivityCategory } from '@/hooks/usePropertyActivities';
import { Card, CardContent } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';

interface PropertyActivitiesTabProps {
  propertyId: string;
}

const activityTypeIcons: Record<string, React.ElementType> = {
  email: Mail,
  call: Phone,
  sms: MessageSquare,
  system: Settings,
};

const activityTypeColors: Record<string, string> = {
  email: 'bg-blue-500/10 text-blue-500',
  call: 'bg-green-500/10 text-green-500',
  sms: 'bg-purple-500/10 text-purple-500',
  system: 'bg-muted text-muted-foreground',
};

const categoryBadgeColors: Record<ActivityCategory, string> = {
  user: 'bg-primary/10 text-primary border-primary/20',
  system: 'bg-muted text-muted-foreground border-border',
  automation: 'bg-warning/10 text-warning border-warning/20',
};

export function PropertyActivitiesTab({ propertyId }: PropertyActivitiesTabProps) {
  const { activities, isLoading } = usePropertyActivities(propertyId);

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-32">
        <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
      </div>
    );
  }

  return (
    <div className="space-y-4">
      <div className="flex items-center justify-between">
        <h3 className="text-lg font-semibold text-foreground">Activity Timeline</h3>
      </div>

      {activities.length === 0 ? (
        <Card className="border-dashed">
          <CardContent className="py-8 text-center text-muted-foreground">
            No activities recorded yet. Activities are logged automatically when property changes occur.
          </CardContent>
        </Card>
      ) : (
        <div className="relative">
          {/* Timeline line */}
          <div className="absolute left-[19px] top-0 bottom-0 w-px bg-border" />
          
          <div className="space-y-4">
            {activities.map((activity) => {
              const Icon = activityTypeIcons[activity.type] || Settings;
              const colorClass = activityTypeColors[activity.type];

              return (
                <div key={activity.id} className="relative flex gap-4">
                  {/* Icon */}
                  <div className={`relative z-10 flex h-10 w-10 shrink-0 items-center justify-center rounded-full ${colorClass}`}>
                    <Icon className="h-4 w-4" />
                  </div>

                  {/* Content */}
                  <Card className="flex-1 bg-card/50 hover:bg-card/80 transition-colors">
                    <CardContent className="p-4">
                      <div className="flex items-start justify-between gap-4">
                        <div className="flex-1 min-w-0">
                          <div className="flex items-center gap-2 flex-wrap">
                            <span className="font-medium text-foreground">
                              {activity.subject || 'Activity'}
                            </span>
                            <Badge 
                              variant="outline" 
                              className={`text-[10px] px-1.5 py-0 ${categoryBadgeColors[activity.category]}`}
                            >
                              {activity.category}
                            </Badge>
                          </div>
                          {activity.body_preview && (
                            <p className="text-sm text-muted-foreground mt-1 line-clamp-2">
                              {activity.body_preview}
                            </p>
                          )}
                        </div>
                        <span className="text-xs text-muted-foreground whitespace-nowrap">
                          {format(new Date(activity.created_at), 'MMM d, h:mm a')}
                        </span>
                      </div>
                    </CardContent>
                  </Card>
                </div>
              );
            })}
          </div>
        </div>
      )}
    </div>
  );
}
