import { motion } from 'framer-motion';
import { CheckCircle2, Circle, Clock, Phone, Mail, Calendar, MoreHorizontal } from 'lucide-react';
import { cn } from '@/lib/utils';
import { Task } from '@/lib/types';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';

interface TasksListProps {
  tasks: Task[];
}

const typeIcons = {
  call: Phone,
  email: Mail,
  meeting: Calendar,
  'follow-up': Clock,
  other: MoreHorizontal,
};

const priorityColors = {
  high: 'text-destructive',
  medium: 'text-warning',
  low: 'text-muted-foreground',
};

export function TasksList({ tasks }: TasksListProps) {
  const formatDate = (date?: Date) => {
    if (!date) return '';
    return new Intl.DateTimeFormat('en-US', {
      month: 'short',
      day: 'numeric',
    }).format(date);
  };

  return (
    <motion.div
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.4, delay: 0.2 }}
      className="rounded-xl border border-border bg-card"
    >
      <div className="flex items-center justify-between border-b border-border p-4">
        <h3 className="font-semibold">Upcoming Tasks</h3>
        <Button variant="ghost" size="sm">
          View all
        </Button>
      </div>

      <div className="divide-y divide-border">
        {tasks.slice(0, 5).map((task, index) => {
          const TypeIcon = typeIcons[task.type];
          return (
            <motion.div
              key={task.id}
              initial={{ opacity: 0, x: -10 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ delay: 0.1 * index }}
              className="flex items-center gap-4 p-4 transition-colors hover:bg-accent/50"
            >
              <button className="text-muted-foreground hover:text-primary transition-colors">
                {task.status === 'completed' ? (
                  <CheckCircle2 className="h-5 w-5 text-success" />
                ) : (
                  <Circle className="h-5 w-5" />
                )}
              </button>

              <div className="flex-1 min-w-0">
                <p className={cn(
                  'font-medium truncate',
                  task.status === 'completed' && 'line-through text-muted-foreground'
                )}>
                  {task.title}
                </p>
                {task.description && (
                  <p className="text-sm text-muted-foreground truncate mt-0.5">
                    {task.description}
                  </p>
                )}
              </div>

              <div className="flex items-center gap-3">
                <Badge variant="secondary" className="gap-1">
                  <TypeIcon className="h-3 w-3" />
                  {task.type}
                </Badge>
                {task.dueDate && (
                  <span className={cn('text-sm', priorityColors[task.priority])}>
                    {formatDate(task.dueDate)}
                  </span>
                )}
              </div>
            </motion.div>
          );
        })}
      </div>
    </motion.div>
  );
}
